import { Pool, QueryResult } from "pg";
const { TIMESERIES_DATABASE_URL } = process.env;

/**
 * Database class for managing PostgreSQL connections.
 * This class implements the Singleton pattern to ensure only one instance exists.
 * It provides methods to execute SQL queries and manage the connection pool.
 * * @class Database
 * @singleton
 * @example
 * const db = Database.getInstance();
 * db.query('SELECT * FROM table_name', [])
 */
export default class Database {
  static instance: Database | null = null;
  pool: Pool | undefined;
  constructor() {
    if (Database.instance) {
      return Database.instance;
    }
    if (!TIMESERIES_DATABASE_URL) {
      throw new Error(
        "Database configuration environment variables are not set"
      );
    }

    this.pool = new Pool({
      connectionString: TIMESERIES_DATABASE_URL,
    });

    Database.instance = this;
  }

  /**
   * Returns the singleton Database instance, creating it if necessary.
   * @return {Database} The singleton Database instance.
   */
  static getInstance() {
    if (!Database.instance) {
      Database.instance = new Database();
    }
    return Database.instance;
  }

  /**
   * Executes a SQL query with parameters.
   * @param {string} sql - The SQL query string.
   * @param {any[]} params - The parameters for the SQL query.
   * @returns {Promise<QueryResult>} The query result promise.
   * @throws {Error} If the database pool is not initialized.
   */
  async query(sql: string, params: any[]): Promise<QueryResult> {
    if (!this.pool) {
      throw new Error("Database pool is not initialized");
    }
    return this.pool.query(sql, params);
  }

  /**
   * Closes the database connection pool and resets the singleton instance.
   * @returns {Promise<void>} A promise that resolves when the pool is closed.
   */
  async close(): Promise<void> {
    await this.pool?.end();
    Database.instance = null;
  }
}
